// cReader - Rex 6000 Chinese Memo Reader
//
// Release version 1.1 - 01/03/2004
//
// Copyright 2003-2004 Jian Ren
//
//
// cReader is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.  
//
// cReader is distributed in the hope that it will be useful, but 
// WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
// GNU General Public License for more details. 
//
// You should have received a copy of the GNU General Public License 
// along with cReader; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// This is only the part of cReader, an updated firmware file contains Chinese character BMPs is required



#include <rex/rex.h>
#include <string.h>

#define MEMO_RECID	1	// 32 bit int - DB record #
#define MEMO_NAME		2	// varstr - Memo name
#define MEMO_TEXT		6	// varstr - text of the memo
#define FIND_NEXT		1	// If record not found, returns next record
#define MEMO_HIS_PAGE	5	//Unknown field used to save bookmark

//#define FONTS_PER_PAGE	315
#define TOTAL_FONTS	8836

#define SIZE_RECIDLIST	8	// # memos per screenful of memo menu
#define SIZE_MEMO_NAME	80
#define MEMO_LIST_LINE_START	24

#define BUTTON_UP		0x6B
#define BUTTON_DOWN	0x6C
#define BUTTON_BACK	0x6D
#define BUTTON_CLOSE	0x6E

#define MEMO_LINE_START		0
#define MEMO_LINE_END		108
#define MEMO_LINE_HEIGHT	12
#define CHINESE_WIDTH		13
#define MEMO_LINE_WIDTH		240
#define MEMO_FIRST_LINE_WIDTH	180
#define FILELIST_ID    0x60	//Event ID for memo list, 0x60-0x69
#define FILELIST_L     MEMO_LINE_HEIGHT
#define FILELIST_W     MEMO_LINE_WIDTH
#define FILELIST_H     (FILELIST_L*SIZE_RECIDLIST)


#define MEMO_BUF_SIZE   		512
#define SIZE_MEMO_BROWSE_HISTORY	512

typedef unsigned char	BYTE;
typedef unsigned int    UINT;
typedef unsigned long 	ULONG;

char *HexChar = 	"0123456789ABCDEF";	// Hex-ASCII lookup table

BYTE UpArrow[] = { 11,9,
	0xff,0xE0, //xxxxxxxxxxx
	0x80,0x20, //x         x
	0x84,0x20, //x    x    x
	0x8E,0x20, //x   xxx   x
	0x8E,0x20, //x   xxx   x
	0x9F,0x20, //x  xxxxx  x
	0xBF,0xA0, //x xxxxxxx x
	0x80,0x20, //x         x
	0xFF,0xE0  //xxxxxxxxxxx
};

BYTE DownArrow[] = { 11,9,
	0xff,0xE0, //xxxxxxxxxxx
	0x80,0x20, //x         x
	0xBF,0xA0, //x xxxxxxx x
	0x9F,0x20, //x  xxxxx  x
	0x8E,0x20, //x   xxx   x
	0x8E,0x20, //x   xxx   x
	0x84,0x20, //x    x    x
	0x80,0x20, //x         x
	0xFF,0xE0  //xxxxxxxxxxx
};


BYTE BackArrow[] = { 11,9,
	0xff,0xE0, //xxxxxxxxxxx
	0x80,0x20, //x         x
	0x88,0x20, //x   x     x
	0xBF,0x20, //x xxxxxx  x
	0x88,0xA0, //x   x   x x
	0x80,0xA0, //x       x x
	0x81,0x20, //x      x  x
	0x82,0x20, //x     x   x
	0xFF,0xE0  //xxxxxxxxxxx
};


BYTE CloseBox[] = { 11,9,
	0xFF,0xE0, //xxxxxxxxxxx
	0xA0,0xA0, //x x     x x
	0x91,0x20, //x  x   x  x
	0x8A,0x20, //x   x x   x
	0x84,0x20, //x    x    x
	0x8A,0x20, //x   x x   x
	0x91,0x20, //x  x   x  x
	0xA0,0xA0, //x x     x x
	0xFF,0xE0  //xxxxxxxxxxx
};

/*
BYTE fontPages[27] = {0x16,0x17,0x18,0x2B,
				0x2C,0x2D,0x2E,0x2F,0x30,0x31,0x32,0x33,
				0x49,0x4D,0x58,0x62,0x78,0x79,0x7a,
     				0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x8b};
*/

//Number of Chinese characters in each page
int fontsPerPage[128] ={0,0,0,0,30,154,143,25,105,69,46,93,12,5,9,19,31,199,115,17,176,238,122,43,10,257,0,0,0,94,61,0,158,112,225,25,27,0,88,67,74,141,20,24,66,285,226,0,0,0,0,0,0,0,0,0,56,24,0,9,10,0,83,23,61,44,20,0,0,11,87,118,8,10,276,72,82,0,111,27,35,0,106,67,76,81,139,44,52,19,5,111,0,90,203,37,49,114,311,270,134,127,0,11,37,18,102,142,174,0,4,160,99,119,65,72,12,151,211,202,78,78,118,264,0,0,0,11};

BYTE ascFontWidth[128] = {
3,1,3,7,5,10,7,1,3,3,5,7,2,3,1,3,
5,4,5,5,5,5,5,5,5,5,1,2,6,7,6,4,
9,6,5,6,6,5,5,6,6,3,4,5,4,7,6,7,
5,7,6,5,5,6,5,9,5,5,5,3,3,3,7,6,
2,5,5,4,5,5,3,5,5,1,2,5,1,7,5,5,
5,5,3,4,3,5,5,7,5,5,4,4,1,4,7,0,
5,0,2,6,4,7,5,5,4,15,5,3,10,0,0,0,
0,2,2,4,4,4,7,9,4,9,4,3,9,0,0,5};

int memoDBHandler;

UINT memoPos;

UINT bufferReadFromPos;
BYTE memoBuffer[MEMO_BUF_SIZE];
UINT bufferPos;
UINT bufferSize;

int RecIdList[SIZE_RECIDLIST];
int currentMemoRecId;

UINT memoPosHistory[SIZE_MEMO_BROWSE_HISTORY];
int memoRecIdHistory[SIZE_MEMO_BROWSE_HISTORY];
int historyPos;
int endOfMemo;

UINT currentMemoPage;

int getTextWidth(BYTE * str) {
	int temp;
	int i;
	int width;
	temp = strlen(str);
	width = 0;
	for (i=0;i<temp;i++) {
		if (str[i]<0xa0)
			width = width + ascFontWidth[str[i]-32] + 1;
	}
	return width;
}

void drawButtons() {
	DsDisplayBitmapDraw(MEMO_FIRST_LINE_WIDTH + 4,0,UpArrow,0);
	DsDisplayBitmapDraw(MEMO_FIRST_LINE_WIDTH + 17,0,DownArrow,0);
	DsDisplayBitmapDraw(MEMO_FIRST_LINE_WIDTH + 34,0,BackArrow,0);
	DsDisplayBitmapDraw(MEMO_FIRST_LINE_WIDTH + 49,0,CloseBox,0);
}

void addButtonEvents() {
	DsEventAdd(MEMO_FIRST_LINE_WIDTH,0,15,MEMO_LINE_HEIGHT,BUTTON_UP,1);
	DsEventAdd(MEMO_FIRST_LINE_WIDTH + 15,0,15,MEMO_LINE_HEIGHT,BUTTON_DOWN,1);
	DsEventAdd(MEMO_FIRST_LINE_WIDTH + 30,0,15,MEMO_LINE_HEIGHT,BUTTON_BACK,1);
	DsEventAdd(MEMO_FIRST_LINE_WIDTH + 45,0,15,MEMO_LINE_HEIGHT,BUTTON_CLOSE,1);
}

void printChinese(BYTE highByte, BYTE lowByte, int x, int y) {
	int offset,pagePos;
	BYTE pageIndex;
	char * ptr;
	offset = (highByte-161)*94 + lowByte-161;
	if (offset<TOTAL_FONTS) {
		pageIndex=4;
		pagePos = fontsPerPage[pageIndex];
		while (offset>=pagePos) {
			offset-=pagePos ;
			pagePos = fontsPerPage[++pageIndex];
		}
		ptr = 0xA000;
		ptr = ptr - (pagePos-offset)*26;
		//Address = 0x2000-pagePos*26 + 0x8000 + offset*26
		DsDisplayBitmapDrawFar(x,y,ptr,pageIndex-4,0);
	}
}

void printStr(int x, int y, char * str) {
	BYTE c1;
	BYTE c2;
	BYTE strLen;
	BYTE strPos;
	BYTE bufPos;
	char printBuf[SIZE_MEMO_NAME];
	int printX;
	strLen = strlen(str);
	strPos=0;
	bufPos=0;
	c1=0;
	c2=0;
	printX = x;
	while (strPos<strLen) {
		c1 = str[strPos++];
		if (c1>0xA0) {
			if (bufPos>0) {
				printBuf[bufPos]=0;
				DsPrintf(printX, y, 0, printBuf);
				printX = printX + getTextWidth(printBuf);
				bufPos=0;
			}
			c2=str[strPos++];
			if (c2>0) {
				printChinese(c1, c2, printX, y);
				printX+=CHINESE_WIDTH;
			}
		}
		else {
			printBuf[bufPos++] = c1;
		}
	}
	if (bufPos>0) {
		printBuf[bufPos]=0;
		DsPrintf(printX, y, 0, printBuf);
	}
}


// Fatal error has occurred: display error code message
// (and optional error return), wait for keypress, then EXIT
void CriticalError()
{
	if (memoDBHandler) DbClose(memoDBHandler);

	DsClearScreen();
	DsPrintf(10,40,16,"Error!");
	DsSleep(300);
	DsAddinTerminate();
}

void locateMemo(int memoRecID) {
	if (DbFindRecord(memoDBHandler, MEMO_RECID, 1, memoRecID, 0) != 1)
		CriticalError();
}

int findNextSubMemo() {
	char *filename;
	char buffer[SIZE_MEMO_NAME];
	int temp;
	int nameLength;
	if (DbReadField(memoDBHandler, MEMO_NAME, &filename)) CriticalError();
	strcpy(buffer, filename);
	buffer[SIZE_MEMO_NAME-1] = 0;
	temp = strlen(buffer);
	if (buffer[temp-1]==')') {
		if (buffer[temp-7] == '(')
			nameLength = temp-7;
		else if (buffer[temp-5] == '(')
			nameLength = temp-5;
		else temp = 0;
	}
	if (temp) {
		temp = DbNextRecord(memoDBHandler);
		if (temp==1) {
			if (DbReadField(memoDBHandler, MEMO_NAME, &filename)) CriticalError();
			if (strncmp(filename, buffer, nameLength))
				temp = 0;
		}
		else
			temp = 0;
	}	
	return temp;
}


//Read one byte from memo, use a cache here, return 0 if no more data in the memo
BYTE ReadMemo() {
	long rlong;
	BYTE result=0;
	if (bufferPos==bufferSize) {
		if (bufferSize==MEMO_BUF_SIZE) {
			//Reload buffer
			if (DbReadText(memoDBHandler, MEMO_TEXT, bufferReadFromPos, 
							MEMO_BUF_SIZE, &bufferSize, memoBuffer))
				CriticalError();
			if (bufferSize) {
				bufferPos = 0;
				bufferReadFromPos = bufferReadFromPos + bufferSize;
				result = memoBuffer[bufferPos++];
			}
			else {
				//End of memo, try sub memos
				bufferSize = MEMO_BUF_SIZE-1;
				bufferPos = bufferSize;
				result = ReadMemo();
			}
		}
		else {
			//End of memo, try the following sub memos
			if (findNextSubMemo(memoDBHandler)) {
				if (DbReadField(memoDBHandler, MEMO_RECID, &rlong)) 
					CriticalError();
				currentMemoRecId = rlong;
				bufferSize = MEMO_BUF_SIZE;
				bufferPos = bufferSize;
				bufferReadFromPos = 0;
				memoPos = 0;
				result = ReadMemo();
			}
		}
	}
	else {
		result = memoBuffer[bufferPos++];
	}
	return result;
}


// MoveFileCursor - to reduce screen repaints, this routine simply 
// UNHIGHLIGHTS the current row, then HIGHLIGHTS the new row
void MoveFileCursor(int old, int new)
{
	DsDisplayBlockReverse(0, MEMO_LIST_LINE_START + FILELIST_L*old,
	                      FILELIST_W, FILELIST_L, 0);
	DsDisplayBlockReverse(0, MEMO_LIST_LINE_START + FILELIST_L*new,
	                      FILELIST_W, FILELIST_L, 0);
}

//Paint the list of memo names on screen for selection, pass in either start or end and the default selection.
//Return start if pass in end, end if pass in start
int displayMemoList(int start, int end, int selected) {
	char *filename;
	int memoIndex; 
	int recIDIndex;
	int returnIndex;
	int temp;
	char buffer[SIZE_MEMO_NAME];
	char strMes[5];
	long rlong;
	memoIndex = 0;
	recIDIndex = 0;
	DsClearScreen();
	DsEventClear();
	drawButtons();
	DsPrintf(40,0,16,"Chinese Reader 1.1");
	DsPrintf(40,10,0,"Copyright 2003-2004 by Jian Ren");
	addButtonEvents();

	DbFindRecord(memoDBHandler, MEMO_NAME, FIND_NEXT, 0x6819, 0);

	do {
		buffer[0] = 0;
		if (DbReadField(memoDBHandler, MEMO_NAME, &filename)) CriticalError();
		strcpy(buffer, filename);
		buffer[SIZE_MEMO_NAME-1] = 0;
		/*
		temp = strlen(buffer);
		if (temp) {
			if (buffer[temp-1]==')') {
				if (buffer[temp-7] == '(') {
					if ((buffer[temp-5]>'1') || (buffer[temp-6]>'0'))
						continue;
					//Remove the numbers followed the name
					buffer[temp-7] = 0;
				}
				else if (buffer[temp-5] == '(') {
					if (buffer[temp-4]>'1') continue;
					buffer[temp-5] = 0;
				}
			}*/
			temp = end - memoIndex;
			if (((!end) && (memoIndex>=start))
						|| ((!start) && (temp<=SIZE_RECIDLIST))) {
				if (end && (temp==SIZE_RECIDLIST-1)) returnIndex = memoIndex; //Return the start
				if (DbReadField(memoDBHandler, MEMO_RECID, &rlong)) CriticalError();
				RecIdList[recIDIndex] = rlong;
				printStr(0, MEMO_LIST_LINE_START + FILELIST_L*recIDIndex, buffer);
				DsEventAdd(0, MEMO_LIST_LINE_START + FILELIST_L*recIDIndex, 
						FILELIST_W, FILELIST_L, FILELIST_ID+recIDIndex, 1);
				recIDIndex++;
			}
			memoIndex++;
		//}
	} while ((recIDIndex<SIZE_RECIDLIST) && (DbNextRecord(memoDBHandler)==1));
	if (!end) returnIndex = memoIndex - 1; //Return the end
	DsDisplayBlockReverse(0, MEMO_LIST_LINE_START + selected*FILELIST_L, FILELIST_W, FILELIST_L, 0);
	return returnIndex;
}

//Print one page memo on the screen
//Return 0 if it's the end of the memo
int printMemoPage(UINT pageNumber, int displayProcessBar) {
	int temp;
	int processBarPos;

	UINT currentPage;

	BYTE c1;
	BYTE c2;
	char str[SIZE_MEMO_NAME];
	int strPos;
	int strWidth;
	int status;
	int i;
	int j;

	strPos = 0;
	strWidth = 0;
	status = 0;
	c1 = 0;
	c2 = 0;

	//Clear the buffer
	bufferSize = MEMO_BUF_SIZE;
	bufferPos = bufferSize;
	//Reset the from position
	bufferReadFromPos = memoPos;

	currentPage = 0;
	processBarPos = 0;
	do {
		if ((pageNumber-currentPage)<SIZE_MEMO_BROWSE_HISTORY) {
			memoPosHistory[historyPos] = memoPos;
			memoRecIdHistory[historyPos] = currentMemoRecId;
			historyPos++;
		}

		i = 0;
		j = MEMO_LINE_START;
		if (currentPage==pageNumber) {
			if (displayProcessBar) {
				while (processBarPos<240)
					DsDisplayLine(processBarPos++,90,0,12,0x1FF);
				DsClearScreen();
			}
			else
				DsDisplayWaitIconDraw(0);
			drawButtons();
			addButtonEvents();
		}
		else {
			if (displayProcessBar) {
				temp = currentPage*239/pageNumber;
				if (temp>processBarPos) {
					while (processBarPos<=temp)
						DsDisplayLine(processBarPos++,90,0,12,0x1FF);
					processBarPos--;
				}
			}
		}
		do {
			//A status machine
			switch(status) {
				case 0:
				//Read from buffer
				c1 = ReadMemo();
				if (c1>0xA0) {
					c2 = c1;
					c1 = ReadMemo();
					if (strPos)
						//Print the previous English at first
						status = 1;
					else
						status = 2;
				}
				else {
					if (c1<=32) {
						if (strPos) {
							status = 1;
							break;
						}
					}
					status = 3;
				}
				break;
			case 1:
				//Print out cached English
				if (j) 
					temp = MEMO_LINE_WIDTH;
				else
					temp = MEMO_FIRST_LINE_WIDTH;
				if (strWidth>(temp-i))
					status = 4;
				else {
					if (currentPage==pageNumber) DsPrintf(i, j, 0, str);
					i = i + strWidth;
					memoPos = memoPos + strPos;
					strPos = 0;
					strWidth = 0;
					if (c2)
						status = 2;
					else
						status = 3;
				}
				break;
			case 2:
				//Print out cached Chinese
				if (j) 
					temp = MEMO_LINE_WIDTH;
				else
					temp = MEMO_FIRST_LINE_WIDTH;
				if (i<=temp-CHINESE_WIDTH) {
					if (currentPage==pageNumber) printChinese(c2, c1, i, j);
					i = i + CHINESE_WIDTH;
					memoPos = memoPos + 2;
					c2 = 0;
					status = 0;
				}
				else
					status = 4;
				break;
			case 3:
				//Handle uncached non Chinese character
				if (c1==10) {
					memoPos ++;
					status = 4;
				}
				else
					if (c1>=32) {
						str[strPos++] = c1;
						str[strPos] = 0;
						strWidth = getTextWidth(str);
						if (strWidth>240) {
							//Go back one character and print out
							strPos--;
							str[strPos] = 0;
							strWidth = getTextWidth(str);
							status = 1;
						}
						else
							status = 0;
					}
					else {
						if (c1) memoPos++;
						status = 0;
					}
				break;
			case 4:
				//New line
				i = 0;
				j = j + MEMO_LINE_HEIGHT;
				if (strPos)
					status = 1;
				else
					if (c2)
						status = 2;
					else
						status = 0;
				break;
			}
		} while ((j<=MEMO_LINE_END) && (c1));
		currentPage++;
	} while (c1 && (currentPage<=pageNumber));
	if (currentPage<=pageNumber) {
		//Nothing left for this page
		DsClearScreen();
		drawButtons();
		addButtonEvents();
	}
	return c1;
}

void displayMemo(int selectedMemo, UINT pageNumber) {
	int temp;
	char str[3];
	DsClearScreen();
	DsEventClear();

	DsDisplayWaitIconDraw(1);
	endOfMemo = 0;

		if (pageNumber==currentMemoPage-1) {
			//Go back, use the history or start from the beginning
			if (historyPos>1) {
				//History available
				historyPos = historyPos - 2;
				memoPos = memoPosHistory[historyPos];
				currentMemoRecId = memoRecIdHistory[historyPos];
				locateMemo(currentMemoRecId);
				endOfMemo = !printMemoPage(0,0);
			}
			else {
				//Start from the beginning
				memoPos = 0;
				historyPos = 0;
				temp = RecIdList[selectedMemo];
				if (currentMemoRecId!=temp) {
					//Use different memo ID
					currentMemoRecId = temp;
					locateMemo(currentMemoRecId);
				}
				if (pageNumber>2) {
					//Display process bar if loop through more than 2 pages, which is slow
					temp = 1;
					DsPrintf(90,79,16,"Loading...");
					DsDisplayLine(0,90,240,0,0x1FF);
					DsDisplayLine(239,90,0,12,0x1FF);
					DsDisplayLine(239,102,-239,0,0x1FF);
					DsDisplayLine(0,102,0,-12,0x1FF);
				}
				else {
					temp = 0;
				}
				endOfMemo = !printMemoPage(pageNumber, temp);
			}
		}
		else {
			if (pageNumber==currentMemoPage+1) {
				//Empty one position if the history was full
				if (historyPos==SIZE_MEMO_BROWSE_HISTORY) {
					for (temp=1;temp<SIZE_MEMO_BROWSE_HISTORY;temp++) {
						memoPosHistory[temp-1] = memoPosHistory[temp];
						memoRecIdHistory[temp-1] = memoRecIdHistory[temp];
					}
					historyPos = SIZE_MEMO_BROWSE_HISTORY - 1;
				}
				endOfMemo = !printMemoPage(0,0);
			}
		}
	currentMemoPage = pageNumber;
}

int main()
{
	MSG msg;
	int flag;
	int previousFlag;
	int selectedMemo;
	int start;
	int end;
	int temp;
	UINT memoPage;
	UINT bookmarkedPage;
	
	start = 0;
	end = 0;
	currentMemoPage = 1;
	memoPos = 0;

	flag = 1;
	selectedMemo = 0;
	memoDBHandler = DbOpen(DBID_MEMO);
	if (!memoDBHandler) CriticalError();

	while (flag) {
		switch(flag) {
			case 1:
				//Displays next screen
				start = end;
				end = displayMemoList(start, 0, selectedMemo);
				previousFlag = 1;
				flag = 4;
				break;
			case 2:
				//Displays previous screen
				end = start;
				selectedMemo = SIZE_RECIDLIST - 1;
				start = displayMemoList(0, end, selectedMemo);
				previousFlag = 2;
				flag = 4;
				break;
			case 3:
				//Print out memo
				displayMemo(selectedMemo, memoPage);
				previousFlag = 3;
				flag = 4;
				break;
			case 4:
				//Handle events
				DsEventMessageGet(&msg);
				switch (msg.message) {
					case MSG_DS_PAINT:
						flag = previousFlag;
						if (flag==4) flag = 1;
						break;
					case MSG_DS_CLOSE:
						flag = 0;
						break;

					case MSG_DS_COMMAND:
						// User touched one of memo on screen
						if (previousFlag!=3 && msg.bCode>=FILELIST_ID && msg.bCode<FILELIST_ID+SIZE_RECIDLIST) {
							temp = selectedMemo;
							selectedMemo = msg.bCode - FILELIST_ID;
							MoveFileCursor(temp, selectedMemo);
							flag = 7;
						}
						else {
							switch(msg.bCode) {
								case BUTTON_UP:
									if (previousFlag<3) {
										selectedMemo = 0;
									}
									flag = 5;
									break;
								case BUTTON_DOWN:
									if (previousFlag<3) {
										selectedMemo = end-start;
									}
									flag = 6;
									break;
								case BUTTON_BACK:
									flag = 8;
									break;
								case BUTTON_CLOSE:
									flag = 0;
									break;
							}
						}
					break;
	    				case MSG_DS_KEY_DOWN:
						switch (msg.sCode) {
							case KEY_TOP_C:
								flag = 0;
								break;
							case KEY_BACK_C:
								flag = 8;
								break;
							case KEY_ENTER_C:
								if (previousFlag!=3) {
									flag = 7;
								}
								break;
							case KEY_UP_C:
								flag = 5;
								break;
							case KEY_DOWN_C:
								flag = 6;
								break;						
						}
					break;
				}
				break;
			case 5:
				//Up button pressed
				if (previousFlag==3) {
					if (memoPage>0) {
						memoPage--;
						flag = 3;
					}
					else
						flag = 4;
				}
				else {
					if (selectedMemo>0) {
						temp = selectedMemo;
						selectedMemo--;
						MoveFileCursor(temp, selectedMemo);
						flag = 4;
					}
					else
						if (start>0) 
							flag = 2;
						else
							flag = 4;
				}
				break;
			case 6:
				//Down button pressed
				if (previousFlag==3) {
					if (!endOfMemo) {
						memoPage++;
						flag = 3;
					}
					else
						flag = 4;
				}
				else {
					if (selectedMemo<(end-start)) {
						temp = selectedMemo;
						selectedMemo++;
						MoveFileCursor(temp, selectedMemo);
						flag = 4;
					}
					else
						if ((end-start)==SIZE_RECIDLIST-1) {
							selectedMemo = 0;
							flag = 1;
						}
						else
							flag = 4;
				}
				break;
			case 7:
				//Print the first page of a memo, or use the bookmark
				currentMemoRecId = RecIdList[selectedMemo];
				locateMemo(currentMemoRecId );
				memoPos = 0;
				//Check if bookmark exists
				if (DbReadField(memoDBHandler, MEMO_HIS_PAGE, &bookmarkedPage))
					bookmarkedPage = 0;
				memoPage = bookmarkedPage;
				currentMemoPage = memoPage+1;
				historyPos = 0;
				flag = 3;
				break;
			case 8:
				//Quit or go back
				if (previousFlag==3) {
					//Back to memo list page
					end = start;
					flag = 1;
					if (currentMemoPage!=bookmarkedPage) {
						//Bookmark
						DsClearScreen();
						DsDisplayWaitIconDraw(1);
						DsPrintf(74,79,16,"Saving Bookmark");
						currentMemoRecId = RecIdList[selectedMemo];
						DbUpdateField(memoDBHandler, MEMO_HIS_PAGE, currentMemoPage);
					}
				}
				else
					flag = 0;
				break;
		}
	}
	DbClose(memoDBHandler);
	DsEventClear();
}
